/**
 * This program is an example from the book "Internet
 * programming with Java" by Svetlin Nakov. It is freeware.
 * For more information: http://www.nakov.com/books/inetjava/
 */
import javax.servlet.*;
import javax.servlet.http.*;
import java.awt.*;
import java.awt.image.*;
import com.sun.image.codec.jpeg.*;
import java.io.*;
import java.util.Date;
import java.text.SimpleDateFormat;

public class ImageCounterServlet extends HttpServlet {
    private static final float JPEG_QUALITY = (float) 0.85;

    private String mStartDate;
    private int mVisitCounter;

    /**
     * Called by the servlet container when the servlet is
     * instantiated. Initializes the counter and start date.
     */
    public void init() {
        Date now = new Date();
        SimpleDateFormat dateFormatter =
            new SimpleDateFormat("d.M.yyyy HH:mm:ss");
        mStartDate = dateFormatter.format(now);
        mVisitCounter = 0;
    }

    /**
     * Creates a graphical image and draws given text with
     * with "Monospaced" yellow font on a red background.
     */
    public BufferedImage createImage(String aMsg) {
        Font font = new Font("Monospaced", Font.BOLD, 16);
        FontMetrics fm = new Canvas().getFontMetrics(font);
        int width = fm.stringWidth(aMsg) + 22;
        int height = fm.getHeight() + 11;
        BufferedImage image = new BufferedImage(
            width, height, BufferedImage.TYPE_INT_RGB);
        Graphics g = image.getGraphics();
        g.setColor(Color.blue);
        g.fillRect(0, 0, width, height);
        g.setColor(Color.yellow);
        g.drawRoundRect(3, 3, width-7, height-7, 15, 15);
        g.setFont(font);
        g.setColor(Color.black);
        g.drawString(aMsg, 11+2, 4 + fm.getAscent()+2);
        g.setColor(Color.yellow);
        g.drawString(aMsg, 11, 4 + fm.getAscent());
        return image;
    }

    /**
     * Called by the servlet container on HTTP GET request.
     * Increases the counter and sends as a response a JPEG
     * image that contains the counter value along with some
     * additional text.
     */
    public void doGet(HttpServletRequest aRequest,
        HttpServletResponse aResponse)
    throws IOException, ServletException {
        String msg;
        synchronized(this) {
            mVisitCounter++;
            msg = "" + mVisitCounter + " visits since " +
                mStartDate;
        }
        BufferedImage image = createImage(msg);
        aResponse.setContentType("image/jpeg");
        OutputStream out = aResponse.getOutputStream();
        JPEGImageEncoder encoder =
            JPEGCodec.createJPEGEncoder(out);
        JPEGEncodeParam jpegParams =
            encoder.getDefaultJPEGEncodeParam(image);
        jpegParams.setQuality(JPEG_QUALITY, false);
        encoder.setJPEGEncodeParam(jpegParams);
        encoder.encode(image);
        out.close();
    }

    /**
     * Called by the servlet container on HTTP POST request.
     * Just delegate to doGet() method.
     */
    public void doPost(HttpServletRequest aRequest,
        HttpServletResponse aResponse)
    throws IOException, ServletException {
        doGet(aRequest, aResponse);
    }
}
