import { z } from 'zod';
import { PostmanAPIClient, ContentType } from '../clients/postman.js';
import {
  IsomorphicHeaders,
  McpError,
  ErrorCode,
  CallToolResult,
} from '@modelcontextprotocol/sdk/types.js';

function asMcpError(error: unknown): McpError {
  const cause = (error as any)?.cause ?? String(error);
  return new McpError(ErrorCode.InternalError, cause);
}

export const method = 'createMonitor';
export const description =
  'Creates a monitor.\n\n**Note:**\n\n- You cannot create monitors for collections added to an API definition.\n- If you do not pass the \\`workspace\\` query parameter, the system creates the monitor in the oldest personal Internal workspace you own.\n';
export const parameters = z.object({
  workspace: z.string().describe("The workspace's ID."),
  monitor: z
    .object({
      name: z.string().describe("The monitor's name."),
      active: z
        .boolean()
        .describe('If true, the monitor is active and makes calls to the specified URL.')
        .default(true),
      notificationLimit: z
        .number()
        .gte(1)
        .lte(99)
        .describe('Stop email notifications after the given number consecutive failures.')
        .optional(),
      collection: z.string().describe("The unique ID of the monitor's associated collection."),
      environment: z
        .string()
        .describe("The unique ID of the monitor's associated environment.")
        .optional(),
      retry: z
        .object({
          attempts: z
            .number()
            .gte(1)
            .lte(2)
            .describe(
              'The number of times to reattempt a monitor run if it fails or errors. This may impact your [monitor usage](https://learning.postman.com/docs/monitoring-your-api/monitor-usage/#view-monitor-usage).'
            )
            .optional(),
        })
        .describe("Information about the monitor's retry settings.")
        .optional(),
      options: z
        .object({
          followRedirects: z.boolean().describe('If true, follow redirects enabled.').optional(),
          requestDelay: z
            .number()
            .gte(1)
            .lte(900000)
            .describe("The monitor's request delay value, in milliseconds.")
            .optional(),
          requestTimeout: z
            .number()
            .gte(1)
            .lte(900000)
            .describe("The monitor's request timeout value, in milliseconds.")
            .optional(),
          strictSSL: z.boolean().describe('If true, strict SSL enabled.').optional(),
        })
        .describe("Information about the monitor's option settings.")
        .optional(),
      schedule: z
        .object({
          cron: z
            .string()
            .describe(
              'The cron expression that defines when the monitor runs. Use standard five-field POSIX cron syntax.\n'
            )
            .optional(),
          timezone: z
            .string()
            .describe(
              "The monitor's [timezone](https://en.wikipedia.org/wiki/List_of_tz_database_time_zones)."
            )
            .optional(),
        })
        .describe("Information about the monitor's schedule."),
      distribution: z
        .array(
          z.object({
            region: z
              .enum([
                'us-east',
                'us-west',
                'ap-southeast',
                'ca-central',
                'eu-central',
                'sa-east',
                'uk',
                'us-east-staticip',
                'us-west-staticip',
              ])
              .describe('The assigned distribution region.')
              .optional(),
          })
        )
        .describe(
          "A list of the monitor's [geographic regions](https://learning.postman.com/docs/monitoring-your-api/setting-up-monitor/#add-regions)."
        )
        .optional(),
      notifications: z
        .object({
          onError: z
            .array(
              z.object({
                email: z
                  .string()
                  .email()
                  .describe('The email address of the user to notify on monitor error.')
                  .optional(),
              })
            )
            .optional(),
          onFailure: z
            .array(
              z.object({
                email: z
                  .string()
                  .email()
                  .describe('The email address of the user to notify on monitor failure.')
                  .optional(),
              })
            )
            .optional(),
        })
        .describe("Information about the monitor's notification settings.")
        .optional(),
    })
    .describe('Information about the monitor.')
    .optional(),
});
export const annotations = {
  title:
    'Creates a monitor.\n\n**Note:**\n\n- You cannot create monitors for collections added to an API definition.\n- If you do not pass the \\`workspace\\` query parameter, the system creates the monitor in the oldest personal Internal workspace you own.\n',
  readOnlyHint: false,
  destructiveHint: false,
  idempotentHint: false,
};

export async function handler(
  args: z.infer<typeof parameters>,
  extra: { client: PostmanAPIClient; headers?: IsomorphicHeaders }
): Promise<CallToolResult> {
  try {
    const endpoint = `/monitors`;
    const query = new URLSearchParams();
    if (args.workspace !== undefined) query.set('workspace', String(args.workspace));
    const url = query.toString() ? `${endpoint}?${query.toString()}` : endpoint;
    const bodyPayload: any = {};
    if (args.monitor !== undefined) bodyPayload.monitor = args.monitor;
    const options: any = {
      body: JSON.stringify(bodyPayload),
      contentType: ContentType.Json,
      headers: extra.headers,
    };
    const result = await extra.client.post(url, options);
    return {
      content: [
        {
          type: 'text',
          text: `${typeof result === 'string' ? result : JSON.stringify(result, null, 2)}`,
        },
      ],
    };
  } catch (e: unknown) {
    if (e instanceof McpError) {
      throw e;
    }
    throw asMcpError(e);
  }
}
