import { z } from 'zod';
import { ContentType } from '../clients/postman.js';
import { McpError, ErrorCode, } from '@modelcontextprotocol/sdk/types.js';
function asMcpError(error) {
    const cause = error?.cause ?? String(error);
    return new McpError(ErrorCode.InternalError, cause);
}
export const method = 'updateWorkspaceGlobalVariables';
export const description = "Updates and replaces a workspace's global [variables](https://learning.postman.com/docs/sending-requests/variables/#variable-scopes). This endpoint replaces all existing global variables with the variables you pass in the request body.";
export const parameters = z.object({
    workspaceId: z.string().describe("The workspace's ID."),
    values: z
        .array(z
        .object({
        key: z.string().describe("The variable's name.").optional(),
        type: z
            .enum(['default', 'secret'])
            .describe('The [type](https://learning.postman.com/docs/sending-requests/variables/#variable-types) of variable.')
            .optional(),
        value: z.string().describe("The variable's value.").optional(),
        enabled: z.boolean().describe('If true, the variable is enabled.').optional(),
        description: z.string().max(512).describe("The variable's description.").optional(),
    })
        .describe('Information about the global variable.'))
        .describe("A list of the workspace's global variables.")
        .optional(),
});
export const annotations = {
    title: "Updates and replaces a workspace's global [variables](https://learning.postman.com/docs/sending-requests/variables/#variable-scopes). This endpoint replaces all existing global variables with the variables you pass in the request body.",
    readOnlyHint: false,
    destructiveHint: false,
    idempotentHint: true,
};
export async function handler(args, extra) {
    try {
        const endpoint = `/workspaces/${args.workspaceId}/global-variables`;
        const query = new URLSearchParams();
        const url = query.toString() ? `${endpoint}?${query.toString()}` : endpoint;
        const bodyPayload = {};
        if (args.values !== undefined)
            bodyPayload.values = args.values;
        const options = {
            body: JSON.stringify(bodyPayload),
            contentType: ContentType.Json,
            headers: extra.headers,
        };
        const result = await extra.client.put(url, options);
        return {
            content: [
                {
                    type: 'text',
                    text: `${typeof result === 'string' ? result : JSON.stringify(result, null, 2)}`,
                },
            ],
        };
    }
    catch (e) {
        if (e instanceof McpError) {
            throw e;
        }
        throw asMcpError(e);
    }
}
