import { z } from 'zod';
import { McpError, ErrorCode, } from '@modelcontextprotocol/sdk/types.js';
function asMcpError(error) {
    const cause = error?.cause ?? String(error);
    return new McpError(ErrorCode.InternalError, cause);
}
export const method = 'deleteSpecFile';
export const description = 'Deletes a file in an API specification. On success, this returns an HTTP \\`204 No Content\\` response.';
export const parameters = z.object({
    specId: z.string().describe("The spec's ID."),
    filePath: z.string().describe('The path to the file.'),
});
export const annotations = {
    title: 'Deletes a file in an API specification. On success, this returns an HTTP \\`204 No Content\\` response.',
    readOnlyHint: false,
    destructiveHint: true,
    idempotentHint: true,
};
export async function handler(args, extra) {
    try {
        const endpoint = `/specs/${args.specId}/files/${args.filePath}`;
        const query = new URLSearchParams();
        const url = query.toString() ? `${endpoint}?${query.toString()}` : endpoint;
        const options = {
            headers: extra.headers,
        };
        const result = await extra.client.delete(url, options);
        return {
            content: [
                {
                    type: 'text',
                    text: `${typeof result === 'string' ? result : JSON.stringify(result, null, 2)}`,
                },
            ],
        };
    }
    catch (e) {
        if (e instanceof McpError) {
            throw e;
        }
        throw asMcpError(e);
    }
}
