﻿// Copyright (c) .NET Foundation and Contributors. Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;

namespace Roslynator.CSharp.Refactorings;

internal static class AddBracesToIfElseRefactoring
{
    public static Task<Document> RefactorAsync(
        Document document,
        IfStatementSyntax ifStatement,
        CancellationToken cancellationToken = default)
    {
        var rewriter = new SyntaxRewriter();

        var newNode = (IfStatementSyntax)rewriter.Visit(ifStatement)
            .WithFormatterAnnotation();

        return document.ReplaceNodeAsync(ifStatement, newNode, cancellationToken);
    }

    private class SyntaxRewriter : CSharpSyntaxRewriter
    {
        private IfStatementSyntax _previousIf;

        public override SyntaxNode VisitIfStatement(IfStatementSyntax node)
        {
            if (node is null)
                throw new ArgumentNullException(nameof(node));

            if (_previousIf?.Equals(node.GetPreviousIf()) != false)
            {
                if (node.Statement?.IsKind(SyntaxKind.Block) == false)
                {
                    IfStatementSyntax ifStatement = node.WithStatement(SyntaxFactory.Block(node.Statement));

                    _previousIf = ifStatement;

                    return base.VisitIfStatement(ifStatement);
                }
                else
                {
                    _previousIf = node;
                }
            }

            return base.VisitIfStatement(node);
        }

        public override SyntaxNode VisitElseClause(ElseClauseSyntax node)
        {
            if (node is null)
                throw new ArgumentNullException(nameof(node));

            if (_previousIf.Equals(node.Parent)
                && node.Statement?.IsKind(SyntaxKind.Block, SyntaxKind.IfStatement) == false)
            {
                return node.WithStatement(SyntaxFactory.Block(node.Statement));
            }

            return base.VisitElseClause(node);
        }
    }
}
