﻿// Copyright (c) .NET Foundation and Contributors. Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System.Collections.Immutable;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.Diagnostics;
using Microsoft.CodeAnalysis.Text;

namespace Roslynator.CSharp.Analysis;

[DiagnosticAnalyzer(LanguageNames.CSharp)]
public sealed class UnnecessaryRawStringLiteralAnalyzer : BaseDiagnosticAnalyzer
{
    private static ImmutableArray<DiagnosticDescriptor> _supportedDiagnostics;

    public override ImmutableArray<DiagnosticDescriptor> SupportedDiagnostics
    {
        get
        {
            if (_supportedDiagnostics.IsDefault)
                Immutable.InterlockedInitialize(ref _supportedDiagnostics, DiagnosticRules.UnnecessaryRawStringLiteral);

            return _supportedDiagnostics;
        }
    }

#if ROSLYN_4_2
    public override void Initialize(AnalysisContext context)
    {
        base.Initialize(context);
        context.RegisterCompilationStartAction(startContext =>
        {
            if (((CSharpCompilation)startContext.Compilation).LanguageVersion >= LanguageVersion.CSharp11)
            {
                startContext.RegisterSyntaxNodeAction(f => AnalyzeStringLiteralExpression(f), SyntaxKind.StringLiteralExpression);
                startContext.RegisterSyntaxNodeAction(f => AnalyzeInterpolatedStringExpression(f), SyntaxKind.InterpolatedStringExpression);
            }
        });
    }

    private static void AnalyzeStringLiteralExpression(SyntaxNodeAnalysisContext context)
    {
        var literalExpression = (LiteralExpressionSyntax)context.Node;

        if (!RawStringLiteralInfo.TryCreate(literalExpression, out RawStringLiteralInfo info))
            return;

        string text = info.Text;

        if (ContainsBackSlashOrQuote(text, info.QuoteCount, text.Length - (info.QuoteCount * 2)))
            return;

        DiagnosticHelpers.ReportDiagnostic(
            context,
            DiagnosticRules.UnnecessaryRawStringLiteral,
            Location.Create(literalExpression.SyntaxTree, new TextSpan(literalExpression.SpanStart + 1, info.QuoteCount - 1)));
    }

    private static void AnalyzeInterpolatedStringExpression(SyntaxNodeAnalysisContext context)
    {
        var interpolatedString = (InterpolatedStringExpressionSyntax)context.Node;

        SyntaxToken startToken = interpolatedString.StringStartToken;

        if (!startToken.IsKind(SyntaxKind.InterpolatedSingleLineRawStringStartToken))
            return;

        foreach (InterpolatedStringContentSyntax content in interpolatedString.Contents)
        {
            if (content is InterpolatedStringTextSyntax interpolatedStringText)
            {
                string text = interpolatedStringText.TextToken.Text;

                if (ContainsBackSlashOrQuoteOrOpenBrace(text, 0, text.Length))
                    return;
            }
        }

        int offset = startToken.ValueText.LastIndexOf('$') + 2;

        DiagnosticHelpers.ReportDiagnostic(
            context,
            DiagnosticRules.UnnecessaryRawStringLiteral,
            Location.Create(interpolatedString.SyntaxTree, new TextSpan(startToken.SpanStart + offset, startToken.Span.Length - offset)));
    }

    private static bool ContainsBackSlashOrQuote(string text, int start, int length)
    {
        for (int pos = start; pos < start + length; pos++)
        {
            switch (text[pos])
            {
                case '\\':
                case '"':
                    return true;
            }
        }

        return false;
    }

    private static bool ContainsBackSlashOrQuoteOrOpenBrace(string text, int start, int length)
    {
        for (int pos = start; pos < start + length; pos++)
        {
            switch (text[pos])
            {
                case '\\':
                case '"':
                case '{':
                    return true;
            }
        }

        return false;
    }
#endif
}
