﻿// Copyright (c) .NET Foundation and Contributors. Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System.Collections.Immutable;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.Diagnostics;

namespace Roslynator.CSharp.Analysis;

[DiagnosticAnalyzer(LanguageNames.CSharp)]
public sealed class FormatSummaryOnMultipleLinesAnalyzer : BaseDiagnosticAnalyzer
{
    private static ImmutableArray<DiagnosticDescriptor> _supportedDiagnostics;

    public override ImmutableArray<DiagnosticDescriptor> SupportedDiagnostics
    {
        get
        {
            if (_supportedDiagnostics.IsDefault)
                Immutable.InterlockedInitialize(ref _supportedDiagnostics, DiagnosticRules.FormatDocumentationSummaryOnMultipleLines);

            return _supportedDiagnostics;
        }
    }

    public override void Initialize(AnalysisContext context)
    {
        base.Initialize(context);

        context.RegisterSyntaxNodeAction(
            f => AnalyzeSingleLineDocumentationCommentTrivia(f),
            SyntaxKind.SingleLineDocumentationCommentTrivia);
    }

    private static void AnalyzeSingleLineDocumentationCommentTrivia(SyntaxNodeAnalysisContext context)
    {
        var documentationComment = (DocumentationCommentTriviaSyntax)context.Node;

        XmlElementSyntax summaryElement = documentationComment.SummaryElement();

        if (summaryElement?.StartTag?.IsMissing == false
            && summaryElement.EndTag?.IsMissing == false
            && summaryElement.IsSingleLine(includeExteriorTrivia: false, trim: false))
        {
            DiagnosticHelpers.ReportDiagnostic(
                context,
                DiagnosticRules.FormatDocumentationSummaryOnMultipleLines,
                summaryElement);
        }
    }
}
