﻿// Copyright (c) .NET Foundation and Contributors. Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System.Collections.Immutable;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.Diagnostics;

namespace Roslynator.CSharp.Analysis;

[DiagnosticAnalyzer(LanguageNames.CSharp)]
public sealed class AvoidUsageOfForStatementToCreateInfiniteLoopAnalyzer : BaseDiagnosticAnalyzer
{
    private static ImmutableArray<DiagnosticDescriptor> _supportedDiagnostics;

    public override ImmutableArray<DiagnosticDescriptor> SupportedDiagnostics
    {
        get
        {
            if (_supportedDiagnostics.IsDefault)
                Immutable.InterlockedInitialize(ref _supportedDiagnostics, DiagnosticRules.AvoidUsageOfForStatementToCreateInfiniteLoop);

            return _supportedDiagnostics;
        }
    }

    public override void Initialize(AnalysisContext context)
    {
        base.Initialize(context);

        context.RegisterSyntaxNodeAction(f => AnalyzeForStatement(f), SyntaxKind.ForStatement);
    }

    private static void AnalyzeForStatement(SyntaxNodeAnalysisContext context)
    {
        var forStatement = (ForStatementSyntax)context.Node;

        if (forStatement.Declaration is null
            && forStatement.Condition is null
            && !forStatement.Incrementors.Any()
            && !forStatement.Initializers.Any()
            && !forStatement.OpenParenToken.ContainsDirectives
            && !forStatement.FirstSemicolonToken.ContainsDirectives
            && !forStatement.SecondSemicolonToken.ContainsDirectives
            && !forStatement.CloseParenToken.ContainsDirectives)
        {
            DiagnosticHelpers.ReportDiagnostic(context, DiagnosticRules.AvoidUsageOfForStatementToCreateInfiniteLoop, forStatement.ForKeyword);
        }
    }
}
