// Copyright (c) .NET Foundation and Contributors. Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System.Collections.Generic;
using System.Collections.Immutable;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Xunit;

namespace Roslynator.CSharp.Analysis.Tests;

public class PatternMatchingVariableDeclarationHelperTests
{
    [Fact]
    public void SingleVariableDesignation()
    {
        VariableDesignationSyntax designation = SyntaxFactory.SingleVariableDesignation(
            SyntaxFactory.Identifier("x")
        );
        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(designation, vars));

        vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(designation, vars));
    }

    [Fact]
    public void ParenthesizedVariableDesignation()
    {
        VariableDesignationSyntax designation = SyntaxFactory.ParenthesizedVariableDesignation(
            SyntaxFactory.SeparatedList(new List<VariableDesignationSyntax>()
            {
                SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("x")),
                SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("y")),
            })
        );

        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(designation, vars));

        vars = new HashSet<string>() { "y" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(designation, vars));

        vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(designation, vars));
    }

    [Fact]
    public void DiscardDesignation()
    {
        VariableDesignationSyntax designation = SyntaxFactory.DiscardDesignation();
        ImmutableHashSet<string> vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(designation, vars));
    }

    [Fact]
    public void NullTest()
    {
        VariableDesignationSyntax designation = null;
        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(designation, vars));
    }

    [Fact]
    public void DeclarationPattern()
    {
        PatternSyntax pattern = SyntaxFactory.DeclarationPattern(
            SyntaxFactory.PredefinedType(SyntaxFactory.Token(SyntaxKind.IntKeyword)),
            SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("x"))
        );
        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));

        vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));
    }

    [Fact]
    public void RecursivePattern_WithPositional()
    {
        PatternSyntax pattern = SyntaxFactory.RecursivePattern(
            SyntaxFactory.IdentifierName("TypeA"),
            positionalPatternClause: SyntaxFactory.PositionalPatternClause(
                SyntaxFactory.SeparatedList(new List<SubpatternSyntax>()
                {
                    SyntaxFactory.Subpattern(
                        SyntaxFactory.DeclarationPattern(
                            SyntaxFactory.PredefinedType(SyntaxFactory.Token(SyntaxKind.IntKeyword)),
                            SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("x"))
                        )
                        ),
                    SyntaxFactory.Subpattern(
                        SyntaxFactory.DeclarationPattern(
                            SyntaxFactory.PredefinedType(SyntaxFactory.Token(SyntaxKind.IntKeyword)),
                            SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("y"))
                        )
                    ),
                })
            ),
            propertyPatternClause: default,
            designation: default
        );

        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));

        vars = new HashSet<string>() { "y" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));

        vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));
    }

    [Fact]
    public void RecursivePattern_WithProperty()
    {
        PatternSyntax pattern = SyntaxFactory.RecursivePattern(
            SyntaxFactory.IdentifierName("TypeA"),
            positionalPatternClause: default,
            propertyPatternClause: SyntaxFactory.PropertyPatternClause(
                SyntaxFactory.SeparatedList(new List<SubpatternSyntax>()
                {
                    SyntaxFactory.Subpattern(
                        SyntaxFactory.NameColon(SyntaxFactory.IdentifierName("PropertyName")),
                        SyntaxFactory.VarPattern(
                            SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("x"))
                        )
                    ),
                })
            ),
            designation: default
        );
        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));

        vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));
    }

    [Fact]
    public void RecursivePattern_WithDesignation()
    {
        PatternSyntax pattern = SyntaxFactory.RecursivePattern(
            SyntaxFactory.IdentifierName("TypeA"),
            positionalPatternClause: default,
            propertyPatternClause: SyntaxFactory.PropertyPatternClause(
                SyntaxFactory.SeparatedList(new List<SubpatternSyntax>()
                {
                    SyntaxFactory.Subpattern(
                        SyntaxFactory.NameColon(SyntaxFactory.IdentifierName("PropertyName")),
                        SyntaxFactory.ConstantPattern(SyntaxFactory.LiteralExpression(SyntaxKind.NumericLiteralExpression, SyntaxFactory.Literal(42)))
                    ),
                })
            ),
           designation: SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("x"))
        );
        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));

        vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));
    }

    [Fact]
    public void VarPattern()
    {
        PatternSyntax pattern = SyntaxFactory.VarPattern(
            SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("x"))
        );
        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));

        vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));
    }

    [Fact]
    public void BinaryPattern()
    {
        PatternSyntax pattern = SyntaxFactory.BinaryPattern(
            SyntaxKind.AndPattern,
            SyntaxFactory.ConstantPattern(SyntaxFactory.LiteralExpression(SyntaxKind.NumericLiteralExpression, SyntaxFactory.Literal(42))),
            SyntaxFactory.ConstantPattern(SyntaxFactory.LiteralExpression(SyntaxKind.NumericLiteralExpression, SyntaxFactory.Literal(99)))
        );
        ImmutableHashSet<string> vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));
    }

    [Fact]
    public void ParenthesizedPattern()
    {
        PatternSyntax pattern = SyntaxFactory.ParenthesizedPattern(
            SyntaxFactory.VarPattern(
                SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("x"))
            )
        );
        ImmutableHashSet<string> vars = new HashSet<string>() { "x" }.ToImmutableHashSet();
        Assert.True(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));

        vars = new HashSet<string>() { "z" }.ToImmutableHashSet();
        Assert.False(PatternMatchingVariableDeclarationHelper.AnyDeclaredVariablesMatch(pattern, vars));
    }
}
