## Redirected/Piped Input

Redirected/piped input will be used as a list of project/solution paths separated with newlines.

## Examples

### Example: Obfuscate all symbols

This is a very simple example that will rename all symbols to random name.

#### Command

```shell
roslynator rename-symbol my.sln
  --match "true"
  --new-name "obfuscate.cs"
```
#### Notes

* We assume that there are files `my.sln` and `obfuscate.cs` in the current directory.

#### `obfuscate.cs`

```cs
using System.IO;
using System.Text.RegularExpressions;
using Microsoft.CodeAnalysis;

public class Obfuscate
{
    public string GetNewName(ISymbol symbol)
    {
        while (true)
        {
            string name = Path.GetRandomFileName();

            if (Regex.IsMatch(name, @"^\D"))
                return Path.GetFileNameWithoutExtension(name) + "xyz";
        }
    }
}
```

### Example: Rename lambda parameters to shorter name

The goal is to rename all lambda parameters that have name `cancellationToken` and are of type `CancellationToken` to `ct`.

#### Command

```shell
roslynator rename-symbol my.sln
  --match "symbol.Name == ""cancellationToken"" && symbol is IParameterSymbol parameterSymbol && parameterSymbol.Type.Name == ""CancellationToken"" && symbol.ContainingSymbol is IMethodSymbol methodSymbol && methodSymbol.MethodKind == MethodKind.AnonymousFunction"
  --new-name " ""ct"" "
  --scope local
```

#### Notes

* We assume that there is file `my.sln` in the current directory.
* The option `--scope local` is not required but it can make the process faster.

### Example: Rename constant so its name contains its value

There is a class that contains various compiler warnings and errors:

```cs
// <auto-generated>

namespace Roslynator.CSharp
{
    internal static class CompilerDiagnosticIdentifiers
    {
        public const string OperatorCannotBeAppliedToOperands = "CS0019";
        public const string CannotApplyIndexingToExpression = "CS0021";
        public const string OperatorCannotBeAppliedToOperand = "CS0023";

        // ...
```

The goal is to add error code to the name of the constant:

```cs
namespace Roslynator.CSharp
{
    internal static class CompilerDiagnosticIdentifiers
    {
        public const string CS0019_OperatorCannotBeAppliedToOperands = "CS0019";
        public const string CS0021_CannotApplyIndexingToExpression = "CS0021";
        public const string CS0023_OperatorCannotBeAppliedToOperand = "CS0023";

        // ...
```

#### Command

```shell
roslynator rename-symbol my.sln
  --match "rename-constants.cs"
  --include-generated-code
  --scope member
```

#### Notes

* We assume that there are files `my.sln` and `rename-constants.cs` in the current directory.
* If the "IsMatch" method and "GetNewName" method are defined both in the same file it is allowed to specified it only once as a value of `--match-from` option.
* Because the code is marked with `auto-generated` tag we have to explicitly allow generated code with `--include-generated-code` switch.
* The option `--scope member` is not required but it can make the process faster.

#### `rename-constants.cs`

```cs
using Microsoft.CodeAnalysis;

public class C
{
    public bool IsMatch(ISymbol symbol)
    {
        return symbol.DeclaredAccessibility == Accessibility.Public
            && symbol is IFieldSymbol fieldSymbol
            && fieldSymbol.IsConst
            && symbol.ContainingType.Name == "CompilerDiagnosticIdentifiers";
    }

    public string GetNewName(ISymbol symbol)
    {
        var fieldSymbol = (IFieldSymbol)symbol;

        var value = (string)fieldSymbol.ConstantValue;

        return value + "_" + symbol.Name;
    }
}
```
