﻿// Copyright (c) .NET Foundation and Contributors. Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System.Collections.Immutable;
using System.Composition;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CodeActions;
using Microsoft.CodeAnalysis.CodeFixes;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Roslynator.CodeFixes;
using static Microsoft.CodeAnalysis.CSharp.SyntaxFactory;

namespace Roslynator.CSharp.CodeFixes;

[ExportCodeFixProvider(LanguageNames.CSharp, Name = nameof(AddBreakStatementToSwitchSectionCodeFixProvider))]
[Shared]
public sealed class AddBreakStatementToSwitchSectionCodeFixProvider : CompilerDiagnosticCodeFixProvider
{
    public override ImmutableArray<string> FixableDiagnosticIds
    {
        get
        {
            return ImmutableArray.Create(
                CompilerDiagnosticIdentifiers.CS0163_ControlCannotFallThroughFromOneCaseLabelToAnother,
                CompilerDiagnosticIdentifiers.CS8070_ControlCannotFallOutOfSwitchFromFinalCaseLabel);
        }
    }

    public override async Task RegisterCodeFixesAsync(CodeFixContext context)
    {
        Diagnostic diagnostic = context.Diagnostics[0];

        SyntaxNode root = await context.GetSyntaxRootAsync().ConfigureAwait(false);

        if (!IsEnabled(diagnostic.Id, CodeFixIdentifiers.AddBreakStatementToSwitchSection, context.Document, root.SyntaxTree))
            return;

        if (!TryFindFirstAncestorOrSelf(root, context.Span, out SwitchSectionSyntax switchSection))
            return;

        CodeAction codeAction = CodeAction.Create(
            "Add 'break' statement",
            ct => RefactorAsync(context.Document, switchSection, ct),
            GetEquivalenceKey(diagnostic));

        context.RegisterCodeFix(codeAction, diagnostic);
    }

    public static Task<Document> RefactorAsync(
        Document document,
        SwitchSectionSyntax switchSection,
        CancellationToken cancellationToken)
    {
        SwitchSectionSyntax newNode = switchSection;

        SyntaxList<StatementSyntax> statements = switchSection.Statements;

        if (statements.Count == 1
            && statements[0].IsKind(SyntaxKind.Block))
        {
            var block = (BlockSyntax)statements[0];
            newNode = newNode.ReplaceNode(block, block.AddStatements(BreakStatement()));
        }
        else
        {
            newNode = switchSection.AddStatements(BreakStatement());
        }

        return document.ReplaceNodeAsync(switchSection, newNode, cancellationToken);
    }
}
