﻿// Copyright (c) .NET Foundation and Contributors. Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System.Collections.Immutable;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.Diagnostics;

namespace Roslynator.CSharp.Analysis;

[DiagnosticAnalyzer(LanguageNames.CSharp)]
public sealed class ValueTypeObjectIsNeverEqualToNullAnalyzer : BaseDiagnosticAnalyzer
{
    private static ImmutableArray<DiagnosticDescriptor> _supportedDiagnostics;

    public override ImmutableArray<DiagnosticDescriptor> SupportedDiagnostics
    {
        get
        {
            if (_supportedDiagnostics.IsDefault)
                Immutable.InterlockedInitialize(ref _supportedDiagnostics, DiagnosticRules.ValueTypeObjectIsNeverEqualToNull);

            return _supportedDiagnostics;
        }
    }

    public override void Initialize(AnalysisContext context)
    {
        base.Initialize(context);

        context.RegisterSyntaxNodeAction(f => AnalyzeEqualsExpression(f), SyntaxKind.EqualsExpression);
        context.RegisterSyntaxNodeAction(f => AnalyzeNotEqualsExpression(f), SyntaxKind.NotEqualsExpression);
    }

    private static void AnalyzeEqualsExpression(SyntaxNodeAnalysisContext context)
    {
        Analyze(context, (BinaryExpressionSyntax)context.Node);
    }

    private static void AnalyzeNotEqualsExpression(SyntaxNodeAnalysisContext context)
    {
        Analyze(context, (BinaryExpressionSyntax)context.Node);
    }

    private static void Analyze(SyntaxNodeAnalysisContext context, BinaryExpressionSyntax binaryExpression)
    {
        ExpressionSyntax left = binaryExpression.Left;

        if (left?.IsMissing == false)
        {
            ExpressionSyntax right = binaryExpression.Right;

            if (right?.Kind() == SyntaxKind.NullLiteralExpression
                && IsStructButNotNullableOfT(context.SemanticModel.GetTypeSymbol(left, context.CancellationToken))
                && !binaryExpression.SpanContainsDirectives())
            {
                DiagnosticHelpers.ReportDiagnostic(
                    context,
                    DiagnosticRules.ValueTypeObjectIsNeverEqualToNull,
                    binaryExpression);
            }
        }
    }

    private static bool IsStructButNotNullableOfT(ITypeSymbol typeSymbol)
    {
        switch (typeSymbol?.TypeKind)
        {
            case TypeKind.Struct:
                return !typeSymbol.IsNullableType();
            case TypeKind.TypeParameter:
                return ((ITypeParameterSymbol)typeSymbol).HasValueTypeConstraint;
            default:
                return false;
        }
    }
}
