﻿// Copyright (c) .NET Foundation and Contributors. Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System.Collections.Immutable;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.Diagnostics;

namespace Roslynator.CSharp.Analysis;

[DiagnosticAnalyzer(LanguageNames.CSharp)]
public sealed class RemoveEmptyElseClauseAnalyzer : BaseDiagnosticAnalyzer
{
    private static ImmutableArray<DiagnosticDescriptor> _supportedDiagnostics;

    public override ImmutableArray<DiagnosticDescriptor> SupportedDiagnostics
    {
        get
        {
            if (_supportedDiagnostics.IsDefault)
                Immutable.InterlockedInitialize(ref _supportedDiagnostics, DiagnosticRules.RemoveEmptyElseClause);

            return _supportedDiagnostics;
        }
    }

    public override void Initialize(AnalysisContext context)
    {
        base.Initialize(context);

        context.RegisterSyntaxNodeAction(f => AnalyzeElseClause(f), SyntaxKind.ElseClause);
    }

    private static void AnalyzeElseClause(SyntaxNodeAnalysisContext context)
    {
        var elseClause = (ElseClauseSyntax)context.Node;

        StatementSyntax statement = elseClause.Statement;

        if (statement?.Kind() != SyntaxKind.Block)
            return;

        var block = (BlockSyntax)statement;

        if (block.Statements.Any())
            return;

        IfStatementSyntax topmostIf = elseClause.GetTopmostIf();

        if (topmostIf.IsParentKind(SyntaxKind.IfStatement))
        {
            var parentIf = (IfStatementSyntax)topmostIf.Parent;

            if (parentIf.Else is not null)
                return;
        }

        if (!elseClause.ElseKeyword.TrailingTrivia.IsEmptyOrWhitespace())
            return;

        SyntaxToken openBrace = block.OpenBraceToken;

        if (!openBrace.LeadingTrivia.IsEmptyOrWhitespace())
            return;

        if (!openBrace.TrailingTrivia.IsEmptyOrWhitespace())
            return;

        if (!block.CloseBraceToken.LeadingTrivia.IsEmptyOrWhitespace())
            return;

        DiagnosticHelpers.ReportDiagnostic(context, DiagnosticRules.RemoveEmptyElseClause, elseClause);
    }
}
